import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Check, User, Video, AlertCircle, Heart, Briefcase, Calendar, Clock, Shield, FileText } from 'lucide-react'
import './BookSession.css'

const BookSession = () => {
  const navigate = useNavigate()
  const [selectedFormat, setSelectedFormat] = useState(null)
  const [selectedTopic, setSelectedTopic] = useState(null)
  const [selectedDate, setSelectedDate] = useState(null)

  const formats = [
    { id: 'in-person', name: 'Лично', icon: User },
    { id: 'video', name: 'Видео', icon: Video }
  ]

  const topics = [
    { id: 'stress', name: 'Стресс', icon: AlertCircle, color: '#ef4444' },
    { id: 'relationships', name: 'Отношения', icon: Heart, color: '#ec4899' },
    { id: 'career', name: 'Карьера', icon: Briefcase, color: '#6366f1' }
  ]

  const generateDates = () => {
    const dates = []
    for (let i = 0; i < 14; i++) {
      const date = new Date()
      date.setDate(date.getDate() + i)
      dates.push(date.toISOString().split('T')[0])
    }
    return dates
  }

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedFormat || !selectedTopic || !selectedDate) {
      alert('Пожалуйста, заполните все поля')
      return
    }

    const sessions = JSON.parse(localStorage.getItem('psychologistSessions') || '{"past": []}')
    const formatName = formats.find(f => f.id === selectedFormat)?.name || selectedFormat
    const topicName = topics.find(t => t.id === selectedTopic)?.name || selectedTopic
    
    const newSession = {
      id: Date.now(),
      date: selectedDate,
      type: formatName,
      topic: topicName,
      rating: null
    }

    sessions.past.push(newSession)
    localStorage.setItem('psychologistSessions', JSON.stringify(sessions))
    
    alert('Сеанс успешно записан!')
    navigate('/')
  }

  return (
    <div className="book-session">
      <div className="hero-booking">
        <div className="hero-booking-left">
          <div className="hero-badge">Новая консультация</div>
          <h1 className="hero-booking-title">Начните свой путь к изменениям</h1>
          <p className="hero-booking-text">Выберите удобный формат встречи, определите тему, которая вас волнует, и назначьте дату. Мы поможем вам найти ответы.</p>
          <div className="hero-features">
            <div className="hero-feature">
              <Check size={20} />
              <span>Гибкое расписание</span>
            </div>
            <div className="hero-feature">
              <Check size={20} />
              <span>Онлайн и офлайн</span>
            </div>
            <div className="hero-feature">
              <Check size={20} />
              <span>Конфиденциально</span>
            </div>
          </div>
        </div>
        <div className="hero-booking-right">
          <div className="hero-visual">
            <div className="visual-circle visual-circle-1"></div>
            <div className="visual-circle visual-circle-2"></div>
            <div className="visual-circle visual-circle-3"></div>
          </div>
        </div>
      </div>

      <form className="booking-form" onSubmit={handleSubmit}>
        <div className="form-section">
          <h2 className="section-title">
            <User className="section-icon" size={24} />
            Выберите формат консультации
          </h2>
          <div className="formats-grid">
            {formats.map(format => {
              const IconComponent = format.icon
              return (
                <button
                  key={format.id}
                  type="button"
                  className={`format-card ${selectedFormat === format.id ? 'selected' : ''}`}
                  onClick={() => setSelectedFormat(format.id)}
                >
                  <div className="format-icon">
                    <IconComponent className="format-icon-img" size={40} />
                  </div>
                  <span className="format-name">{format.name}</span>
                </button>
              )
            })}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <FileText className="section-icon" size={24} />
            Выберите тему консультации
          </h2>
          <div className="topics-grid">
            {topics.map(topic => {
              const IconComponent = topic.icon
              return (
                <button
                  key={topic.id}
                  type="button"
                  className={`topic-card ${selectedTopic === topic.id ? 'selected' : ''}`}
                  onClick={() => setSelectedTopic(topic.id)}
                  style={{ '--topic-color': topic.color }}
                >
                  <div className="topic-icon">
                    <IconComponent className="topic-icon-img" size={32} />
                  </div>
                  <span className="topic-name">{topic.name}</span>
                </button>
              )
            })}
          </div>
        </div>

        <div className="form-section">
          <h2 className="section-title">
            <Calendar className="section-icon" size={24} />
            Выберите дату сеанса
          </h2>
          <div className="dates-scroll">
            {generateDates().map(date => {
              const dateObj = new Date(date)
              const isToday = date === new Date().toISOString().split('T')[0]
              return (
                <button
                  key={date}
                  type="button"
                  className={`date-button ${isToday ? 'today' : ''} ${selectedDate === date ? 'selected' : ''}`}
                  onClick={() => setSelectedDate(date)}
                >
                  <div className="date-day">{dateObj.getDate()}</div>
                  <div className="date-weekday">{dateObj.toLocaleDateString('ru-RU', { weekday: 'short' })}</div>
                </button>
              )
            })}
          </div>
        </div>

        <button type="submit" className="submit-button" disabled={!selectedFormat || !selectedTopic || !selectedDate}>
          Записаться на консультацию
        </button>
      </form>

      <div className="why-choose-section">
        <h2 className="why-choose-title">Почему выбирают нас</h2>
        <div className="why-choose-grid">
          <div className="why-card">
            <div className="why-card-number">01</div>
            <h3>Конфиденциальность</h3>
            <p>Ваши данные под защитой, сеансы проходят анонимно</p>
          </div>
          <div className="why-card">
            <div className="why-card-number">02</div>
            <h3>Опытные специалисты</h3>
            <p>Сертифицированные психологи с многолетним стажем</p>
          </div>
          <div className="why-card">
            <div className="why-card-number">03</div>
            <h3>Гибкий формат</h3>
            <p>Онлайн и офлайн консультации в удобное время</p>
          </div>
        </div>
      </div>

      <div className="info-section">
        <div className="info-section-header">
          <h2 className="info-title">Полезная информация</h2>
        </div>
        <div className="info-grid">
          <div className="info-card">
            <div className="info-card-icon-wrapper">
              <Clock className="info-card-icon" size={32} />
            </div>
            <h3>Длительность сеанса</h3>
            <p>Стандартная консультация длится 50-60 минут. Возможны расширенные сеансы по договоренности.</p>
          </div>
          <div className="info-card">
            <div className="info-card-icon-wrapper">
              <FileText className="info-card-icon" size={32} />
            </div>
            <h3>Подготовка к сеансу</h3>
            <p>Подготовьте список вопросов и тем, которые хотите обсудить. Для онлайн-сеансов проверьте технику.</p>
          </div>
          <div className="info-card">
            <div className="info-card-icon-wrapper">
              <Shield className="info-card-icon" size={32} />
            </div>
            <h3>Конфиденциальность</h3>
            <p>Все сеансы строго конфиденциальны. Мы гарантируем полную анонимность и защиту ваших данных.</p>
          </div>
        </div>
      </div>
    </div>
  )
}

export default BookSession

